
globals [
  ;; model parameters
  number-of-house
  number-of-cars
  max-wtp                       ;; maximal wtp of healthy cars for a time unit of parking
  decrease-in-wtp-per-unit      ;; how much the current wtp deacrease per tick when the car is parked
  number-of-parking-places      ;; how many parking places are there
  mov-avg-length                ;; length of moving averages
  ;; aggregate statistics
  ;; -- current data in the last tick
  current-profit                ;; current owner's profit in the tick
  current-surplus               ;; current consumers' surplus in the tick
  current-usage-percents        ;; current per cents of parking lot used in the tick
  ;; -- list of data for ticks within a day
  parking-firm-profits          ;; profits of the owner within a day
  surpluses-of-parking-cars     ;; surpluses of the parking car within a day
  usage-percents-list           ;; how much is the parking lot used within a day in per cents
                   ;; storing data about the price evolution within a day
  ;; -- daily data
  last-price                    ;; average price within the last day
  ;; -- list of daily data
  daily-profit-list             ;; daily profits of the parking lots' owner
  daily-surplus-list            ;; daily consumers' surpluses
  daily-total-surplus-list      ;; daily total surpluses (total welfares)
  daily-usage-list              ;; daily average of how much is the parking lot used by all cars

]

;;object and its variables
breed [ cars car ]
cars-own [
  parked          ;; true if the car is parked, false otherwise
  wtp             ;; the car's general WTP, constant over time
  actual-wtp      ;; the car's current WTP, decreasing when parked
  not-moved       ;; true if the car has not move in the period
]
breed [houses house]

;; SETUP


to startup
  setup
end

;; setup
to setup
  clear-all
  if not random-start? [ random-seed initial-random-seed ]
  setup-globals
  setup-cars
  setup-road
  reset-ticks
end

;; setup the global variables
to setup-globals
  ;; setup the model parameters
  set number-of-house 1
  set number-of-cars 10
  set max-wtp 100
  set decrease-in-wtp-per-unit 10
  set number-of-parking-places 8
  set mov-avg-length 30
  ;; setup the aggregate statistics
  set surpluses-of-parking-cars ( list )
  set parking-firm-profits ( list )
  set current-profit 0
  set current-surplus 0
  set usage-percents-list ( list )
  set daily-profit-list ( list )
  set daily-surplus-list ( list )
  set daily-total-surplus-list ( list )
  set daily-usage-list ( list )
  set price 10
end

;; setup the cars
to setup-cars
  create-cars number-of-cars [
    set wtp random ( max-wtp + 1 )
    set actual-wtp wtp
    set parked false
    set heading 90
    set shape "car"
    set size 0.75
    set color blue
    set not-moved false
    setxy 0 0
    if ( any? patches with [ ( pycor = 0 ) and ( count cars-here = 0 ) ] ) [
      move-to one-of patches with [ ( pycor = 0 ) and ( count cars-here = 0 ) ]
    ]
  ]

end

;; setup the road and the parking lots
to setup-road
  create-turtles 1[
  set shape "house"
  set color red
  set size 1
  setxy 15 3
  ]
  let pp ( round ( ( max-pxcor - number-of-parking-places ) / 2 ) )
  ask patches with [(pycor = 1) and (pxcor >= pp) and (pxcor < ( pp + number-of-parking-places ))] [
    set pcolor white
  ]
end


;; major simulations loop
to go
  ;; decrease the actual wtp of parked cars
  ask cars with [ parked ] [ set actual-wtp ( actual-wtp - decrease-in-wtp-per-unit ) ]
  ;; major move: unpark, move, park
  move
  ;; restore wtp of the cars that left the parking lot and cannot park now
  ask cars with [ ( not parked ) and ( ( [ pcolor ] of patch-at 0 -1 ) = black ) ]
  [ set actual-wtp ( wtp + ifelse-value ( TRUE ) [ actual-wtp ] [ 0 ] ) ]
  store-data
  tick
end

to move
  ask cars with [ not parked ] [ set not-moved true ]
  loop [
    ;; stop if no not-moved cars
    if ( not any? cars with [ ( not parked ) and not-moved and ( not any? cars-on patch-ahead 1 ) ] ) [ stop ]
    ;; leave the parking lot
    ask cars with [ parked and ( actual-wtp < price ) and ( not any? cars-on patch-at 0 -1 ) ]
    [
      set heading 180
      fd 1
      set heading 90
      set not-moved true
      set parked false

    ]
    ;; move forward on the road
    let s ( cars with [ ( not parked ) and not-moved and ( not any? cars-on patch-ahead 1 ) ] )
    if any? s
    [
      ask one-of s
      [
        fd 1
        set not-moved false
      ]
    ]
    ;; enter the parking lot
    ask cars with [ ( [ pcolor ] of patch-at 0 1 = white ) and ( not any? cars-on patch-at 0 1 ) and
    ( actual-wtp >= price) ]
    [
      set heading 0
      fd 1
      set heading 180
      set not-moved false
      set parked true
    ]
  ]
end

;; store the aggregate data into global variables
to store-data
  ;; current data in the tick
  let current-usage ( count cars with [ parked ] )
  set current-profit ( current-usage * price )
  let current-wtp ( sum [ actual-wtp ] of cars with [ parked ] )
  set current-surplus ( current-wtp - current-profit )
  set current-usage-percents ( current-usage / number-of-parking-places * 10 )
  ;; list with current data in ticks within a day
  set surpluses-of-parking-cars ( lput current-surplus surpluses-of-parking-cars )
  if ( ( length surpluses-of-parking-cars ) > world-width ) [
    set surpluses-of-parking-cars ( but-first surpluses-of-parking-cars )
  ]
  set parking-firm-profits ( lput current-profit parking-firm-profits )
  if ( ( length parking-firm-profits ) > world-width ) [
    set parking-firm-profits ( but-first parking-firm-profits )
  ]
  set usage-percents-list ( lput current-usage-percents usage-percents-list )
  if ( ( length usage-percents-list ) > world-width ) [
    set usage-percents-list ( but-first usage-percents-list )
  ]

  ;; at the end of the day
  if ( ( ticks > 0 ) and ( ( ticks + 1 ) mod world-width = 0 ) )
  [
    set daily-profit-list ( lput daily-profit daily-profit-list )
    if ( ( length daily-profit-list ) > mov-avg-length ) [
      set daily-profit-list ( but-first daily-profit-list )
    ]
    set daily-surplus-list ( lput daily-surplus daily-surplus-list )
    if ( ( length daily-surplus-list ) > mov-avg-length ) [
      set daily-surplus-list ( but-first daily-surplus-list )
    ]
    set daily-total-surplus-list ( lput daily-total-surplus daily-total-surplus-list )
    if ( ( length daily-total-surplus-list ) > mov-avg-length ) [
      set daily-total-surplus-list ( but-first daily-total-surplus-list )
    ]

    set daily-usage-list ( lput daily-mean-usage-percents daily-usage-list )
    if ( ( length daily-usage-list ) > mov-avg-length ) [
      set daily-usage-list ( but-first daily-usage-list )
    ]

  ]
end

;; reports lots owner's daily profit
to-report daily-profit
  report sum parking-firm-profits
end

;; reports consumers' daily surplus
to-report daily-surplus
  report sum surpluses-of-parking-cars
end

;; reports daily total surplus
to-report daily-total-surplus
  report ( daily-profit + daily-surplus )
end

;; reports how much is the parking lot used within a day in per cents
to-report daily-mean-usage-percents
  report mean usage-percents-list
end

;; Moving average values
;;;;;;;;;;;;;;;;;;;;;;;;;;

;; reports the moving average of the daily profits of the parking lots' owner
to-report average-daily-profit
  report mean daily-profit-list
end

;; reports the moving average of the daily consumers' surpluses
to-report average-daily-surplus
  report mean daily-surplus-list
end

;; reports the moving average of the daily total surpluses
to-report average-daily-total-surplus
  report mean daily-total-surplus-list
end


;; reports the moving average of how much is the parking lot used by all cars
to-report average-usage
  report mean daily-usage-list
end
@#$#@#$#@
GRAPHICS-WINDOW
5
90
933
145
-1
-1
23.0
1
10
1
1
1
0
1
1
1
0
39
0
1
1
1
1
ticks
10.0

BUTTON
5
10
75
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
80
10
150
43
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
155
10
225
43
step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
5
50
225
83
price
price
0
200
10.0
1
1
NIL
HORIZONTAL

PLOT
5
175
480
385
Welfare
days
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"profit" 1.0 0 -8053223 true "" "if ( ( ticks > 0 ) and ( ticks mod world-width = 0 ) ) [ plot average-daily-profit ]"
"total" 1.0 0 -15040220 true "" "if ( ( ticks > 0 ) and ( ticks mod world-width = 0 ) ) [ plot average-daily-total-surplus ]"

MONITOR
5
520
170
565
day
floor ( ticks / world-width )
0
1
11

MONITOR
175
570
340
615
last day profit
last daily-profit-list
0
1
11

MONITOR
175
520
340
565
mean profit
average-daily-profit
0
1
11

SWITCH
485
45
705
78
show-daily-values?
show-daily-values?
0
1
-1000

SWITCH
485
8
705
41
random-start?
random-start?
0
1
-1000

INPUTBOX
710
10
930
85
initial-random-seed
15.0
1
0
Number

MONITOR
345
520
510
565
mean consumer's surplus
average-daily-surplus
0
1
11

MONITOR
515
520
680
565
mean total surplus
average-daily-total-surplus
0
1
11

MONITOR
345
570
510
615
last day cs
last daily-surplus-list
0
1
11

MONITOR
515
570
680
615
last day total surplus
last daily-total-surplus-list
0
1
11

MONITOR
515
620
680
665
mean total usage %
daily-mean-usage-percents
0
1
11

PLOT
5
390
480
510
Price
days
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"price   " 1.0 0 -16777216 true "" "if ( ( ticks > 0 ) and ( ticks mod world-width = 0 ) ) [ plot last-price ]"

@#$#@#$#@
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.3.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
