import numpy as np

MAP_SIZE = 41
slam_map = np.zeros((MAP_SIZE, MAP_SIZE))  
trash_map = np.zeros((MAP_SIZE, MAP_SIZE))  

def update_position(sensor_data, current_position):
    x, y, heading = current_position
    heading = float(heading)

    print(f"Current position: x={x}, y={y}, heading={heading}")
    print(f"Sensor data: {sensor_data}")

    
    update_map(sensor_data, x, y, heading)

    
    nearest_trash = find_nearest_trash(x, y)
    if nearest_trash:
        target_x, target_y = nearest_trash
        angle_to_trash = np.degrees(np.arctan2(target_y - y, target_x - x))
        heading = angle_to_trash % 360

        
        if is_path_clear(x, y, heading, sensor_data):
            x += np.cos(np.radians(heading))
            y += np.sin(np.radians(heading))
        else:
            
            heading = (heading + 45) % 360
    else:
        
        if is_path_clear(x, y, heading, sensor_data):
            x += np.cos(np.radians(heading))
            y += np.sin(np.radians(heading))
        else:
            heading = (heading + 45) % 360  

    
    x = max(0, min(MAP_SIZE - 1, x))
    y = max(0, min(MAP_SIZE - 1, y))

    print(f"Updated position: x={x}, y={y}, heading={heading}")
    return [x, y, heading, slam_map.tolist(), trash_map.tolist()]



def find_nearest_trash(robot_x, robot_y):
 
    global trash_map
    trash_indices = np.argwhere(trash_map == 1)  
    if len(trash_indices) == 0:
        return None  

    
    distances = [
        (tx, ty, np.sqrt((tx - robot_x)**2 + (ty - robot_y)**2))
        for tx, ty in trash_indices
    ]
    nearest = min(distances, key=lambda t: t[2])  
    return [nearest[0], nearest[1]]

def is_path_clear(robot_x, robot_y, heading, sensor_data):

    global slam_map

   
    distance = sensor_data[0]
    target_x = robot_x + np.cos(np.radians(heading))
    target_y = robot_y + np.sin(np.radians(heading))

    
    if 0 <= int(round(target_x)) < MAP_SIZE and 0 <= int(round(target_y)) < MAP_SIZE:
        clear = slam_map[int(round(target_x)), int(round(target_y))] == 0
        print(f"Path to x={target_x}, y={target_y} clear: {clear}")  
        return clear
    print(f"Path to x={target_x}, y={target_y} is out of bounds.")
    return False


def update_map(sensor_data, robot_x, robot_y, robot_heading):
    """
    Обновляет карту препятствий и обнаруживает мусор.
    """
    global slam_map, trash_map

    print(f"Updating map at position: x={robot_x}, y={robot_y}, heading={robot_heading}")
    for i, distance in enumerate(sensor_data):
        angle = robot_heading + i * 45
        angle = np.radians(angle)

        obs_x = robot_x + distance * np.cos(angle)
        obs_y = robot_y + distance * np.sin(angle)

        obs_x = int(round(obs_x))
        obs_y = int(round(obs_y))

        if 0 <= obs_x < MAP_SIZE and 0 <= obs_y < MAP_SIZE:
            if slam_map[obs_x, obs_y] == 0:
                trash_map[obs_x, obs_y] = 1  # Если не препятствие, считаем это мусором
                print(f"Detected trash at: x={obs_x}, y={obs_y}")
            slam_map[obs_x, obs_y] = 1


def remove_trash(x, y):
    """
    Удаляет мусор с карты мусора.
    
    :param x: Координата X мусора.
    :param y: Координата Y мусора.
    """
    global trash_map
    trash_map[int(round(x)), int(round(y))] = 0

def set_trash_positions(positions):
    """
    Устанавливает начальные координаты мусора на карте мусора (trash_map).
    
    :param positions: Список координат мусора [[x1, y1], [x2, y2], ...].
    """
    global trash_map
    for x, y in positions:
        trash_map[int(round(x)), int(round(y))] = 1

def get_map():
    """
    Возвращает текущую карту.
    
    :return: Карта в виде списка списков.
    """
    return slam_map.tolist()


